<?php
/**
 * Tatsu Builder Update via Licence Checker
 */

if ( ! class_exists( 'TatsubuilderUpdateChecker' ) ) {
	class TatsubuilderUpdateChecker {

		protected $api_url = 'https://tatsubuilder.com';
		protected $updater_check;
		protected $license_key;
		protected $item_id;

		public function __construct() {
			$this->license_key = trim( get_option( 'tatsu_license_key', '' ) );
			$this->item_id = intval( get_option( 'tatsu_license_item_id' ) );

			add_action( 'init', array( $this, 'check_plugin_update' ) );
			add_action( 'tatsu_pro_license_check', array( $this, 'check_license' ) );
			add_action( 'admin_notices', array( $this, 'admin_notices' ) );
			add_action( 'tatsu_license_tpl', array( $this, 'license_status' ) );
			add_filter( 'auto_update_plugin', array( $this, 'disable_auto_update' ), 10, 2 );
		}

		public function check_plugin_update() {
			// register cron event
			if ( ! wp_next_scheduled( 'tatsu_pro_license_check' ) ) {
				wp_schedule_event( time(), 'daily', 'tatsu_pro_license_check' );
			}

			// To support auto-updates, this needs to run during the wp_version_check cron job for privileged users.
			$doing_cron = defined( 'DOING_CRON' ) && DOING_CRON;
			if ( ! current_user_can( 'manage_options' ) && ! $doing_cron ) {
				return;
			}

			if ( ! $this->license_key ) {
				return;
			}

			require_once dirname( __FILE__ ) . '/class-tatsu-check.php';

			$this->updater_check = new \Tatsu_Check(
				$this->api_url,
				TATSU_PRO_PLUGIN_FILE,
				array(
					'version' => TATSU_PRO_VERSION,
					'license' => $this->license_key,
					'item_id' => $this->item_id,
					'author' => 'Brand Exponents',
					'url' => home_url()
				)
			);
		}

		public function check_license() {
			global $wp_version;

			$status = $this->get_license_data( 'license' );

			$api_params = array(
				'edd_action' => 'check_license',
				'license' => $this->license_key,
				'item_id' => $this->item_id,
				'url'       => home_url()
			);
		
			// Call the custom API.
			$response = wp_remote_post( $this->api_url, array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );
		
			if ( is_wp_error( $response ) ) {
				return false;
			}
		
			$license_data = (array) json_decode( wp_remote_retrieve_body( $response ) );

			update_option( 'tatsu_pro_license_data', $license_data );

			if ( 'valid' !== $status && 'valid' === $license_data['license'] ) {
				$dismissed_notices = get_option( 'tatsu_admin_dismiss_notices', array() );
				if ( $dismissed_notices && is_array( $dismissed_notices ) ) {
					$notices = array( 'tatsu-pro-license-renewal-notice', 'tatsu-pro-license-expired-notice' );
					update_option( 'tatsu_admin_dismiss_notices', array_diff( $dismissed_notices, $notices ) );
				}
			}
		}

		public function license_status() {
			$status = $this->get_license_data( 'license' );
			$expire = $this->get_license_data( 'expires' );

			if ( ! $status || ! $expire ) {
				return;
			}
			
			if ( 'valid' === $status ) {
				$status_code = '<span style="color: green;">' . ucwords( $status ) . '</span>';
			} else {
				$status_code = '<span style="color: red;">' . ucwords( str_replace( '_', ' ', $status ) ) . '</span>';
			}

			echo '<h2 style="margin-top: 30px;">' . esc_html( 'License Status: ', 'tatsu-pro' ) . $status_code . '</h2>';

			if ( 'valid' === $status ) {
				if ( 'lifetime' === $expire ) {
					$validity = __( 'Lifetime', 'tatsu-pro' );
				} else {
					$diff = abs( strtotime( $expire ) - strtotime( 'now' ) );
					$years = floor( $diff / ( 365 * 60 * 60 * 24 ) );
					$months = floor( ( $diff - $years * 365*60*60*24 ) / ( 30*60*60*24 ) );
					$days = floor( ( $diff - $years * 365*60*60*24 - $months*30*60*60*24 ) / ( 60*60*24 ) );
					$validity = sprintf( "%d years, %d months, %d days", $years, $months, $days );
				}

				echo '<h4 style="margin: 0;">' . esc_html( 'License Validity: ', 'tatsu-pro' ) . $validity . '</h4>';
			}
		}

		public function admin_notices() {
			$status = $this->get_license_data( 'license' );
			$expire = $this->get_license_data( 'expires' );

			if ( ! $status || ! $expire ) {
				return;
			}

			if ( 'lifetime' === $expire ) {
				return;
			}

			$dismissed_notices = get_option( 'tatsu_admin_dismiss_notices', array() );
			if ( ! is_array( $dismissed_notices ) ) {
				$dismissed_notices = array();
			}

			$renew_url = add_query_arg( [
				'edd_license_key' => $this->license_key,
				'download_id' => $this->item_id,
			], $this->api_url . '/checkout' );

			if ( 'valid' === $status ) {
				$timestamp = strtotime( $expire );
				$now = strtotime( 'now' );
				$gap = $timestamp - $now;
				$day_in_seconds = 30 * DAY_IN_SECONDS;

				if ( $gap < $day_in_seconds && ! in_array( 'tatsu-pro-license-renewal-notice', $dismissed_notices ) ) { ?>
					<div id="tatsu-pro-license-renewal-notice" class="notice notice-warning is-dismissible tatsu-admin-notice">
						<p>
							<span class="dashicons dashicons-warning" style="vertical-align: middle; margin: -3px 3px 0 0;"></span> 
							<?php printf(
								esc_html__( 'Your %s license is about to expire, only 1 month left. Renew your License to continue to receive updates & support.', 'tatsu-pro' ), 
								'<strong>' . __( 'Tatsu Pro', 'tatsu-pro' ) . '</strong>'
							); ?> 
							<a href="<?php echo esc_url( $renew_url ); ?>" target="_blank" class="button button-primary" style="margin-left: 8px;"><?php esc_html_e( 'Renew Now', 'wp-auto-republish' ); ?></a>
						</p>
					</div>
					<?php
				}
			} else {
				$last_shown = get_option( 'tatsu_pro_expired_notice_last_shown' );
				$now = strtotime( '-3 months' );

				if ( $last_shown && $now > $last_shown ) {
					if ( ( $key = array_search( 'tatsu-pro-license-expired-notice', $dismissed_notices ) ) !== false ) {
						unset( $dismissed_notices[ $key ] );
						update_option( 'tatsu_admin_dismiss_notices', $dismissed_notices );
						$dismissed_notices = get_option( 'tatsu_admin_dismiss_notices', array() );
					}
				}

				if ( ! in_array( 'tatsu-pro-license-expired-notice', $dismissed_notices ) ) {
					update_option( 'tatsu_pro_expired_notice_last_shown', time(), false ); ?>
					<div id="tatsu-pro-license-expired-notice" class="notice notice-error is-dismissible tatsu-admin-notice">
						<p>
							<span class="dashicons dashicons-warning" style="vertical-align: middle; margin: -3px 3px 0 0;"></span> 
							<?php printf(
								esc_html__( 'Your %s license has expired!!! You will not receive any critical updates & support, to resume', 'tatsu-pro' ), 
								'<strong>' . __( 'Tatsu Pro', 'tatsu-pro' ) . '</strong>'
							); ?> 
							<a href="<?php echo esc_url( $renew_url ); ?>" target="_blank" class="button button-primary" style="margin-left: 5px;"><?php esc_html_e( 'Renew Now', 'wp-auto-republish' ); ?></a>
						</p>
					</div>
					<?php
				}
			}
		}

		public function disable_auto_update( $update, $plugin ) {
			if ( 'tatsu-pro/tatsu-pro.php' === $plugin->plugin && 'valid' !== $this->get_license_data( 'license' ) ) {
				return false;
			}
		
			return $update;
		}

		public function get_license_data( $field = false ) {
			$data = get_option( 'tatsu_pro_license_data', array() );
			if ( empty( $data ) || ! is_array( $data ) ) {
				return '';
			}

			if ( $field ) {
				return isset( $data[ $field ] ) ? $data[ $field ] : '';
			}

			return $data;
		}
	}
}