<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       http://example.com
 * @since      1.0.0
 *
 * @package    Tatsu_Pro
 * @subpackage Tatsu_Pro/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Tatsu_Pro
 * @subpackage Tatsu_Pro/includes
 * @author     Your Name <email@example.com>
 */
class Tatsu_Pro {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Tatsu_Pro_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $tatsu_pro    The string used to uniquely identify this plugin.
	 */
	protected $tatsu_pro;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'TATSU_PRO_VERSION' ) ) {
			$this->version = TATSU_PRO_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->tatsu_pro = 'tatsu-pro';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Tatsu_Pro_Loader. Orchestrates the hooks of the plugin.
	 * - Tatsu_Pro_i18n. Defines internationalization functionality.
	 * - Tatsu_Pro_Admin. Defines all hooks for the admin area.
	 * - Tatsu_Pro_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * Helper functions
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/helpers/helper.php';

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-tatsu-pro-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-tatsu-pro-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-tatsu-pro-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-tatsu-pro-public.php';

		$this->loader = new Tatsu_Pro_Loader();

		$this->register_modules();

	}

		/**
	 * Check Oshine and Spyro Modules are exist or not.
	 * If not exists, then only load them.
	 *
	 * @since    1.0.1
	 * @access   private
	 */
	private function register_modules() {

		if ( \is_tatsu_pro_valid() ) {

			$this->loader->add_filter( 'tatsu_module_categories', $this, 'register_categories' );

			/**
			 * The class responsible for add more modules and functionality
			 * to the plugin.
			 */
			if ( ! \class_exists( 'Oshine_Modules' ) ) {
				require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/oshine-modules/oshine-modules.php';
			}

			/**
			 * The class responsible for  add more modules and functionality
			 * to the plugin.
			 */
			if ( ! \class_exists( 'Spyro_Modules' ) ) {
				require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/spyro-modules/spyro-modules.php';

				/**
				 * Create table for spyro forms
				 */
				if ( \function_exists( 'check_and_make_tables_for_tatsu_forms' ) ) {
					\check_and_make_tables_for_tatsu_forms();
				}
			}

			/**
			 * The class responsible for  add more modules and functionality
			 * to the plugin.
			 */
			foreach ( glob( plugin_dir_path( dirname( __FILE__ ) ) . 'includes/modules/*.php' ) as $module ) {
				require_once $module;
			}
		}

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Tatsu_Pro_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Tatsu_Pro_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Tatsu_Pro_Admin( $this->get_tatsu_pro(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Tatsu_Pro_Public( $this->get_tatsu_pro(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_tatsu_pro() {
		return $this->tatsu_pro;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Tatsu_Pro_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

	/**
	 * Register Tatsu Module Categories.
	 *
	 * @since 1.0.2
	 */
	public function register_categories( $categories ) {
		$new_categories = [
			'tab' => __( 'Tabs', 'tatsu-pro' ),
			'forms' => __( 'Forms', 'tatsu-pro' ),
			'slider' => __( 'Slider', 'tatsu-pro' ),
			'animation' => __( 'Animation', 'tatsu-pro' ),
			'title' => __( 'Title', 'tatsu-pro' ),
			'pricing' => __( 'Pricing', 'tatsu-pro' ),
			'posts' => __( 'Posts', 'tatsu-pro' ),
		];

		return array_merge( $categories, $new_categories );
	}

}
